import com.sap.idm.vds.*;
import java.util.HashMap;
import java.util.Iterator;
import com.sap.idm.vds.connector.samplefile.impl.*;
import java.util.*;


/**
 * This connector class acts as a file browser.
 * Several actions are allowed to be taken:
 * 		- Search -> Permits to browse through the file system.
 * 					This one can be in several levels
 * 		- Modify -> Permits to modify the content of a file.
 * 					A couple of options are available for this
 * 		- Add -> Used to add files and directories in the file system
 * 		- Delete -> Used to delete files and directories
 *
 * For a more complete description of each action look at the method description related with each one of them, please
 * 
 * @author I054742
 *
 */
public class FileBrowserConnector extends com.sap.idm.vds.AbstractOperation {
	
	/* Constants used to get specify information from a DN */
	private final int ROOT_NAME = 0;
	private final int PATH = 1;
	private final int FILE_TYPE = 2;
	private final int FILE_NAME = 3;
	private final int ROOT_VALUE = 4;
	private final int PATH_UNTIL_FILE = 5;
	private final int NUMBER_OF_FIELDS = 6;
	
	/* Constants for diverse options in search */
	private final int BASE_SEARCH = 0;
	
	/* Constants for diverse options in modify */
	private final int DELETE_OPTION_MODIFY = 1;
	
	
	/**
	 * Creates a file path from the LDAP starting point parameter and gives the root of the virtual tree
	 * @param param	-> Set of LDAP parameters
	 * @param searchOpt	-> Search option
	 * @param searchOperation -> Indicates if we called this method from a search operation
	 * @param dn_dir -> Name used in DN to distinguish directories
	 * @param dn_file -> Name used in DN to distinguish normal files
	 * @return -> A vector
	 * 				- 0 -> Root of the virtual tree
	 * 				- 1 -> Starting point
	 * 				- 2 -> Type of objective (file or directory)
	 * 				- 3 -> Name of the file or directory in the tail of the path
	 * 				- 4 -> Value of the root of the virtual tree
	 * 				- 5 -> Includes all the path except the last file or directory in it
	 */
	private Vector DNtoOS (String dn, boolean searchOperation, String dn_dir, String dn_file) {
		
		/* Splits the original DN in tokens using as delimiter ',' */
        String [] tokenizer = dn.split(",");
        ArrayList list = new ArrayList();
        Vector res = new Vector();
        String [] strs = new String[NUMBER_OF_FIELDS];
        for (int i=0; i<NUMBER_OF_FIELDS; i++) strs[i]="";
        
        /* For any operation which is not a search operation we need a file or directory, not only the root of the virtual tree */
        if (searchOperation==false && tokenizer.length==1) return null;
        
        /* Checks if each token is right in the DN */
        for (int i=0; i<tokenizer.length; i++) {
        	String token = tokenizer[i];
        	if (token.indexOf("=")>=0) {
        		String [] aux = token.split("=");
        		/* If something illegal is finding in the DN then a null value is returned */
        		if (i<tokenizer.length-1 && aux[0].equalsIgnoreCase(dn_file)==false && aux[0].equalsIgnoreCase(dn_dir)==false) return null;
        		else if (i>0 && aux[0].equalsIgnoreCase(dn_file)) return null;
        		
        		/* Indicates if the target is a file */
        		if (aux[0].equalsIgnoreCase(dn_file)) strs[FILE_TYPE]=dn_file;
        		/* Used to memorize which is the root of the virtual tree */
        		if (i==tokenizer.length-1) {strs[ROOT_NAME]=aux[0]; strs[ROOT_VALUE]=aux[1];}
        		/* Name of the file or directory in the tail of the path */
        		if (i==0) strs[FILE_NAME]=aux[1];
        		list.add(aux[1]);
        	}
        	else return null;
        }
        
        /* Constructs the file path */
        String path = list.get(list.size()-1)+":\\";
        for (int i=list.size()-2; i>=0; i--) {
        	if (i<list.size()-2) path+="\\";
        	/* Includes the complete path except the last file or directory in it */
        	if (i==0) strs[PATH_UNTIL_FILE]=path;
        	path+=list.get(i);
        }
        /* Complete path */
        strs[PATH]=path;
        for (int i=0; i<6; i++) res.add(strs[i]);
        
        return res;
	}
	
	
	/**
	 * Transforms an operating system file path in the corresponding DN
	 * @param path -> OS path
	 * @param aType -> Type of the file (normal file or directory) described by the file path
	 * @param root -> Name of the root of the DN
	 * @param dn_dir -> Name used in DN to distinguish directories
	 * @param dn_file -> Name used in DN to distinguish normal files
	 * @return -> The DN
	 */
	private String OStoDN (String path, String aType, String root, String dn_dir, String dn_file) {
		
		/* Splits the file path according to the OS delimiter for the file path */
        String [] tokenizer = path.split("\\\\");
        String dn = "";
        
        /* Creates the DN according to the type of file (normal file or directory) */
        if (tokenizer.length>1) {
        	if (aType.equalsIgnoreCase("directory")==true) dn+=dn_dir+"="+tokenizer[tokenizer.length-1]+",";
        	else dn+=dn_file+"="+tokenizer[tokenizer.length-1]+",";
        }
        
        for (int i=tokenizer.length-2; i>=1; i--) {
        	dn+=dn_dir+"="+tokenizer[i]+",";
        }
        
        dn+=(root+"="+((tokenizer[0]).split(":"))[0]);
        
        return dn;
	}
	
	
	/**
	 * Finds a MVDModAttrValues object in a vector of this type according to its attribute name
	 * @param v -> Vector of MVDModAttrValues
	 * @param name -> Attribute name of the desired MVDModAttrValues object
	 * @return -> Either the desired object if it is in the vector, or 'null' otherwise 
	 */
	private Vector searchAttrByName (Vector v, String name) {
		
		MVDModAttrValues res = new MVDModAttrValues();
		for (int i=0; i<v.size(); i++) {
			String attrName = ((MVDModAttrValues)v.get(i)).getAttrName();
			if (attrName.equalsIgnoreCase(name)) res.add(v.get(i));
		}
		if (res.size()>0) return res;
		else return null;
	}
	
	
	public MVDOperationResult  bind  (MVDHashMap param) {
        MVDOperationResult result = new MVDOperationResult();
        // --- Place your java implementation here
        return result; 
    }

	
	/**
	 * Realizes the search operation according to one of the next options:
	 * 		- Base search -> Gives a set of attributes of a file
	 * 		- One level search -> Gives the set of adjoining files and directories under a directory
	 * 		- Subtree search -> Gives the complete subtree of files and directories under a directory
	 * @param param	-> set of LDAP parameters
	 * @return -> The result indicating success, or indicating error if some error occurred
	 */
    public MVDSearchResults search (MVDHashMap param) {
    	
    	MVDSearchResults result = new MVDSearchResults(); 
        
    	/* Gets the name to refer directories in DN */
        String dn_dir = (String)param.getDSMandatoryParameter("DN_DIRECTORY");
        /* Gets the name to refer normal files in DN */
        String dn_file = (String)param.getDSMandatoryParameter("DN_FILE");
        /* Gets the starting point parameter */
        String aSp = (String)param.getLdapMandatoryParameter("STARTINGPOINT");
        /* Gets the search type */
        int searchOpt = ((Integer)param.getLdapMandatoryParameter("OPSUBTYPE")).intValue();
        /* Gets the filter */
        String filter = (String)param.getLdapMandatoryParameter("URLFILTER");
        
        /* Checks if the filter syntax is correct */
        String checkFilter = GenericOperations.cleanTrailingAndLeadingSpaces(filter);
        if (checkFilter.endsWith(")")==false || checkFilter.startsWith("(")==false ||
        		checkFilter.indexOf("=")<0) {
        	result.set(87, "Invalid filter format");
        	return result;
        }
        
        /* Transforms the starting point in LDAP format to a understandable format for the operating system */
        Vector v = this.DNtoOS(aSp,true,dn_dir,dn_file);
        if (v==null) {
        	result.setError(34, "Error in DN format for this operation");
        	return result;
        }
        /* Gets the complete path where to do the search from */
        String path = (String)v.get(PATH);
        /* If path is null means that something was not correct in the DN syntax */
        if (path==null) {
        	result.setError(21, "Null Path");
        	return result;
        }
        
        /* Checks if it is expected a directory as starting point */
        boolean isDNDirectory = (((String)v.get(FILE_TYPE)).equalsIgnoreCase(dn_file))==false;
        
        /* Memorizes name of the root of the DN */
        String root_name = (String)v.get(ROOT_NAME);
        
        /* Does the search */
		FileSearch fs = new FileSearch();
		int resCode = fs.search(path, filter, searchOpt, isDNDirectory);
		if (resCode!=FileConstants.SUCCESS) {
			result.setError(FileConstants.LDAP_CODES[resCode], FileConstants.MESSAGES[resCode]);
			return result;
		}
		
		if (isDNDirectory==false && searchOpt!=FileConstants.BASE_SEARCH) {
			result.setError(21, "This search operation is only possible for directories");
			return result;
		}
		
		HashMap res = fs.getSearchResult();
		
		/* If the option is base search then ... */
        if (searchOpt==BASE_SEARCH) {
        	if (res.size()>0) {
	        	/* ... the entry is a file attribute and one of its values */
	        	MVDSearchResultEntry e = new  MVDSearchResultEntry();
	        	e.setDn(aSp);
	        	/* Adds to the entry the set of attributes and values received from the call to the search 
        		 * method of the API file browser connector */
	        	for (Iterator it=res.keySet().iterator(); it.hasNext(); ) {
	        		String key = (String)it.next();
	        		String val = (String)res.get(key);
	        		e.setAttrValue(key,true,val);
	        	}
	        	/* Adds the entry to the result */
	    		result.add(e);
        	}
        }
        else {
			/* Creates the result to be returned */
			for (Iterator it = res.keySet().iterator(); it.hasNext(); ) {
	            
				String key = (String)it.next();
	            HashMap aType = (HashMap)res.get(key);
	            MVDSearchResultEntry e = new  MVDSearchResultEntry();
	            String fileType = (String)aType.get("objectclass");
	            /* ... the entry is a DN indicating a path file or directory */
	            e.setDn(this.OStoDN(key,fileType,root_name,dn_dir,dn_file));
	            e.setAttrValue("objectclass",true,(String)aType.get("objectclass"));
	            String ident = null;
	            if (aType.containsKey("dir")) ident = "dir";
	            else ident = "file";
	            e.setAttrValue(ident,true,(String)aType.get(ident));
	            result.add(e);
			}
        }
		
		result.setOK();
        return result; 
    }
    
    
    /**
     * Modifies the content of a file. The three possibilities are:
     * 		- Add at the end of a file some extra content
     * 		- Replace the current content by a new one
     * @param param	-> set of LDAP parameters
	 * @return -> Result to indicate that everything runs properly
     */
    public MVDOperationResult modify (MVDHashMap param) {
    	
        MVDOperationResult result = new MVDOperationResult();
        
        /* Gets the name to refer directories in DN */
        String dn_dir = (String)param.getDSMandatoryParameter("DN_DIRECTORY");
        /* Gets the name to refer normal files in DN */
        String dn_file = (String)param.getDSMandatoryParameter("DN_FILE");
        /* Gets the starting point */
        String dn = (String)param.getLdapMandatoryParameter("DN");
        /* Transforms the starting point in LDAP format to an understandable format for the operating system */
        Vector v = (Vector)(this.DNtoOS(dn,false,dn_dir,dn_file));
        /* Some error in parsing the DN */
        if (v==null) {
        	result.setError(34, "Error in DN format for this operation");
        	return result;
        }     
        /* Gets the path where the new file or directory must be added */
        String path = (String)v.get(PATH);
        /* Gets if the objective is a file or not */
        boolean isDNDirectory = (((String)v.get(FILE_TYPE)).equalsIgnoreCase(dn_file))==false;
        
        /* This operation is not allowed for directories */
        if (isDNDirectory) {
        	result.setError(34, "This operation is not allowed of this type of entries");
        	return result;
        }
        
        /* Gets the new content and the modification type */
        Vector vData = (Vector)param.getLdapMandatoryParameter("DATA");
        Vector attrs = searchAttrByName(vData,"content");
        String content = "";
        int modType=-1;
        if (attrs!=null && attrs.size()>0) {
        	modType = ((MVDModAttrValues)attrs.firstElement()).getModType();
        	if (modType==DELETE_OPTION_MODIFY) {
            	result.setError(21, "Delete option is not permitted for modify operations");
            	return result;
            }
        	if (attrs.size()>1) {
        		result.setError(21,"The attribute does not allow multivalue data "+vData);
        		return result;
        	}
        	if (((MVDModAttrValues)attrs.firstElement()).size()>0) {
        		content = (String)((MVDModAttrValues)attrs.firstElement()).firstElement();
        	}
        }
        
        /* Makes the modification */
        FileModify fileMod = new FileModify();
        int resCode = fileMod.modify(path,content,modType,isDNDirectory);
        if (resCode!=FileConstants.SUCCESS) {
			result.setError(FileConstants.LDAP_CODES[resCode], FileConstants.MESSAGES[resCode]);
			return result;
		}
        
        /* Indicates that the operation was realized successfully */
        result.setOK();
        return result;
    }
    
    
    /**
     * Adds a new file
     * @param param	-> Set of LDAP parameters
	 * @return -> Result to indicate that everything run properly
     */
    public MVDOperationResult add (MVDHashMap param) {
   
        MVDOperationResult result = new MVDOperationResult();
        
        /* Gets the name to refer directories in DN */
        String dn_dir = (String)param.getDSMandatoryParameter("DN_DIRECTORY");
        /* Gets the name to refer normal files in DN */
        String dn_file = (String)param.getDSMandatoryParameter("DN_FILE");
        /* Gets the directory where the file must be added in LDAP DN format */
        String dn = (String)param.getLdapMandatoryParameter("DN");
        /* Transforms the LDAP format to an understandable format for the operating system */
        Vector v = (Vector)(this.DNtoOS(dn,false,dn_dir,dn_file));
        /* Some error in parsing the DN */
        if (v==null) {
        	
        	result.setError(34, "Error in DN format for this operation");
        	return result;
        }     
        
        /* Gets the path where the new file or directory must be added */
        String path_until_file = (String)v.get(PATH_UNTIL_FILE);
        /* Gets the name of the file or directory to be added */
        String fileName = (String)v.get(FILE_NAME);
        /* Gets if the type of file is a normal file */
        boolean isDNDirectory = (((String)v.get(FILE_TYPE)).equalsIgnoreCase(dn_file))==false;
        
        Object oData = param.getLdapMandatoryParameter("DATA");
        HashMap hData = null;
        if (oData==null) hData = new HashMap();
        else hData = (HashMap)oData;
        
        /* Either gets the content and the file name for files or the directory name for directories */
        String content = "";
        if (isDNDirectory==false) {
        	/* Case for files... */
        	/* Checks the existence of the mandatory parameter that gives the file name */
        	if (hData.containsKey("file")==false) {
    			result.setError(22,"Missed the mandatory attribute with name \"file\" "+hData);
    			return result;
    		}
        	/* Checks if the file name is a single attribute */
        	Vector vFile = (Vector)hData.get("file");
        	if (vFile.size()!=1) {
        		result.setError(21,"The attribute \"file\" must be a single attribute "+hData);
    			return result;
        	}
        	/* Gets the value of the attribute that gives the attribute file name */
        	String fileNameAtt = (String)vFile.get(0);
        	/* Checks if the value of the attribute file name is the same as the related one in the DN */
        	if (fileNameAtt.equalsIgnoreCase(fileName)==false) {
        		result.setError(21,"The attribute \"file\" must have the same value as the related part in the DN "+hData);
        		return result;
        	}
        	/* Checks if the attribute content exists */
        	if (hData.size()==2) {
        		if (hData.containsKey("content")==false) {
        			result.setError(21,"Only two attributes are accepted for this type of entry: \"file\" and \"content\" "+hData);
        			return result;
        		}
        	}
        	/* In this case error in the number of sent attributes */
        	else if (hData.size()<1 || hData.size()>2) {
        		result.setError(21,"Only two attributes are accepted for this type of entry: \"file\" and \"content\" "+hData);
        		return result;
        	}
        	/* Gets the content of the file */
        	Vector vContent = (Vector)hData.get("content");
        	if (vContent!=null) {
        		if (vContent.size()!=1) {
        			result.setError(21,"The attribute \"content\" can only have one value "+vContent);
        			return result;
        		} 
        		else content = (String)vContent.get(0);
        	}
        }
        else {
        	/* Case for directories... */
        	/* Checks the existence of the mandatory parameter that gives the directory name */
        	if (hData.containsKey("dir")==false) {
    			result.setError(22,"Missed the mandatory attribute with name \"dir\" "+hData);
    			return result;
    		}
        	/* Checks if the directory name is a single attribute */
        	Vector vDir = (Vector)hData.get("dir");
        	if (vDir.size()!=1) {
        		result.setError(21,"The attribute \"dir\" must be a single attribute "+hData);
    			return result;
        	}
        	/* Gets the value of the attribute that gives the attribute directory name */
        	String dirNameAtt = (String)vDir.get(0);
        	/* Checks if the value of the attribute directory name is the same as the related one in the DN */
        	if (dirNameAtt.equalsIgnoreCase(fileName)==false) {
        		result.setError(21,"The attribute \"dir\" must have the same value as the related part in the DN "+hData);
        		return result;
        	}
        	/* Checks there is no more attribute than the legal one */
        	if (hData.size()!=1) {
	        	result.setError(21,"Only one attribute is accepted for this type of entry: \"dir\" "+hData);
	    		return result;
        	}
        }
        
        /* Adds the file */
        FileAdd file = new FileAdd();
        int resCode = file.add(path_until_file,fileName,content,isDNDirectory);
        if (resCode!=FileConstants.SUCCESS) {
			result.setError(FileConstants.LDAP_CODES[resCode], FileConstants.MESSAGES[resCode]);
			return result;
		}
        
        result.setOK();
        return result;
    }
    
    
    /**
     * Deletes a file
     * @param param	-> Set of LDAP parameters
	 * @return -> Result to indicate that everything run properly
     */
    public MVDOperationResult  delete  (MVDHashMap param) {
    	
        MVDOperationResult result = new MVDOperationResult();
        
        /* Gets the name to refer directories in DN */
        String dn_dir = (String)param.getDSMandatoryParameter("DN_DIRECTORY");
        /* Gets the name to refer normal files in DN */
        String dn_file = (String)param.getDSMandatoryParameter("DN_FILE");
        /* Gets the file path on LDAP DN format */
        String dn = (String)param.getLdapMandatoryParameter("DN");
        /* Transforms the LDAP format to an understandable format for the operating system */
        Vector v = (Vector)(this.DNtoOS(dn,false,dn_dir,dn_file));
        /* Some error in parsing the DN */
        if (v==null) {
        	
        	result.setError(34, "Error in DN format for this operation");
        	return result;
        }
        /* Gets the path where the new file or directory must be added */
        String path = (String)v.get(PATH);
        /* Gets if the type of file is a normal file */
        boolean isDNDirectory = (((String)v.get(FILE_TYPE)).equalsIgnoreCase(dn_file))==false;
        
        /* Deletes the file */
        FileDelete fileDelete = new FileDelete();
        int resCode = fileDelete.delete(path,isDNDirectory);
        if (resCode!=FileConstants.SUCCESS) {
			result.setError(FileConstants.LDAP_CODES[resCode], FileConstants.MESSAGES[resCode]);
			return result;
		}
        
        result.setOK();
        return result; 
    }
    
    
    public MVDOperationResult  compare  (MVDHashMap param) {
        MVDOperationResult result = new MVDOperationResult();
        // --- Place your java implementation here
        return result; 
    }

    public MVDOperationResult initialize (MVDHashMap param) {
        MVDOperationResult result = new MVDOperationResult(); 
        // --- Place your java implementation here
        return result; 
    }
    
    public MVDOperationResult terminate (MVDHashMap param) {
        MVDOperationResult result = new MVDOperationResult();
        // --- Place your java implementation here
        return result; 
    }
}